/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/* Lots and lots of debugging information */

Components.utils.import("resource://gre/modules/utils.js");

var Bindwood = {
    bookmarksService: Cc["@mozilla.org/browser/nav-bookmarks-service;1"]
        .getService(Ci.nsINavBookmarksService),
    uuidService: Cc["@mozilla.org/uuid-generator;1"]
        .getService(Ci.nsIUUIDGenerator),
    annotationService: Cc["@mozilla.org/browser/annotation-service;1"]
        .getService(Ci.nsIAnnotationService),
    consoleService: Cc["@mozilla.org/consoleservice;1"]
        .getService(Ci.nsIConsoleService),
    historyService: Cc["@mozilla.org/browser/nav-history-service;1"]
        .getService(Ci.nsINavHistoryService),
    ioService: Cc["@mozilla.org/network/io-service;1"]
        .getService(Ci.nsIIOService),
    envService: Cc["@mozilla.org/process/environment;1"]
        .getService(Ci.nsIEnvironment),
    directoryService: Cc["@mozilla.org/file/directory_service;1"]
        .getService(Ci.nsIProperties),
    windowService: Cc["@mozilla.org/appshell/window-mediator;1"]
        .getService(Ci.nsIWindowMediator),

    push: 'ENABLED', // Start off enabled
    annotationKey: "bindwood/uuid",
    uuidItemIdMap: {},

    // Debugging/Console Behavior
    writeMessage: function(aMessage) {
        // convenience method for logging. Way better than alert()s.
        if (Bindwood.envService.exists("BINDWOOD_DEBUG")) {
            Bindwood.consoleService.logStringMessage("Bindwood: " + aMessage);
        }
    },

    writeError: function(aMessage, e) {
        // This should fire whether we're in DEBUG or not
        Bindwood.consoleService.logStringMessage("Bindwood: " + aMessage + " message: '" + e.message + "', reason: '" + e.reason + "', description: '" + e.description + "', error: '" + e.error + "'");
    },

    extractProfileName: function(path) {
        // We want the last part of the profile path
        // ('default' for '/home/zbir/.mozilla/firefox/ixlw0nvl.default')
        // For profiles that have not been created via the Profile Manager,
        // we just take the last path segment as the profile name.
        //
        // Actually, there's a degenerate case, where the last segment
        // doesn't have the Firefox-provided random string:
        //   '/home/zbir/canonical.com' => 'com'
        // But as I said, this is maybe degenerate.

        var segments = path.split('/');
        var possible_profile = segments[segments.length - 1];
        var first_period = possible_profile.indexOf('.');
        if (first_period == -1) {
            return possible_profile; // no periods in the last segment, return as is
        } else {
            return possible_profile.substr(first_period + 1);
        }
    },

    // Setup the environment and go
    init: function() {
        // Start the process and de-register ourself
        // http://forums.mozillazine.org/viewtopic.php?f=19&t=657911&start=0
        // It ensures that we're only running that code on the first window.

        Bindwood.currentProfile = Bindwood.extractProfileName(Bindwood.directoryService.get('ProfD', Ci.nsIFile).path);

        if(Bindwood.windowService.getEnumerator("").getNext() == window) {
            Bindwood.getCouchEnvironment(Bindwood.startProcess);
        }
    },

    getCouchEnvironment: function(continueFunction) {
        // find the desktop Couch port number by making a D-Bus call
        // we call D-Bus by shelling out to a bash script which calls
        // it for us, and writes the port number into a temp file

        // find OS temp dir to put the tempfile in
        // https://developer.mozilla.org/index.php?title=File_I%2F%2FO#Getting_special_files
        var tmpdir = Cc["@mozilla.org/file/directory_service;1"]
                     .getService(Ci.nsIProperties)
                     .get("TmpD", Ci.nsIFile);
        // create a randomly named tempfile in the tempdir
        var tmpfile = Cc["@mozilla.org/file/local;1"]
                     .createInstance(Ci.nsILocalFile);
        tmpfile.initWithPath(tmpdir.path + "/desktopcouch." + Math.random());
        tmpfile.createUnique(tmpfile.NORMAL_FILE_TYPE, 0600);

        // find the D-Bus bash script, which is in our extension folder
        var MY_ID = "bindwood@ubuntu.com";
        var em = Cc["@mozilla.org/extensions/manager;1"].
            getService(Ci.nsIExtensionManager);
        var couchdb_env_script = em.getInstallLocation(MY_ID).getItemFile(MY_ID, "couchdb_env.sh");
        // create an nsILocalFile for the executable
        var nsifile = Cc["@mozilla.org/file/local;1"]
                     .createInstance(Ci.nsILocalFile);
        nsifile.initWithPath(couchdb_env_script.path);

        // create an nsIProcess2 to execute this bash script
        var process = Cc["@mozilla.org/process/util;1"]
            .createInstance(Ci.nsIProcess2);
        process.init(nsifile);

        // Run the process, passing the tmpfile path
        var args = [tmpfile.path];
        process.runAsync(args, args.length, {
            observe: function(process, finishState, unused_data) {
                // If the script exists cleanly, we should have a file
                // containing the port couch is running on as well as
                // the various OAuth tokens necessary to talk to it.
                if (finishState == "process-finished") {
                    // read temp file to find couch environment
                    // https://developer.mozilla.org/en/Code_snippets/File_I%2f%2fO#Reading_from_a_file
                    var environment;
                    var fstream = Cc["@mozilla.org/network/file-input-stream;1"].
                        createInstance(Ci.nsIFileInputStream);
                    var cstream = Cc["@mozilla.org/intl/converter-input-stream;1"].
                        createInstance(Ci.nsIConverterInputStream);
                    fstream.init(tmpfile, -1, 0, 0);
                    cstream.init(fstream, "UTF-8", 0, 0);
                    let (str = {}) {
                      cstream.readString(-1, str); // read the whole file and put it in str.value
                      environment = str.value;
                    };
                    cstream.close(); // this closes fstream
                    environment = environment.replace(/^\s\s*/, '').replace(/\s\s*$/, '');
                } else {
                    // If we fail, we should just return
                    Bindwood.writeMessage("D-Bus port find failed");
                }
                tmpfile.remove(false);

                if (environment == 'ENOCOUCH') {
                    // No Couch environment found. Just spit out a
                    // message and return, stopping Bindwood from
                    // doing anything further.
                    Bindwood.writeError("No suitable Couch environment found. Not proceeding.", e);
                } else if (environment) {
                    // If we don't have a Couch environment, don't bother
                    // trying to fall back on the system CouchDB.
                    continueFunction(environment);
                } else { // No environment found at all
                  return;
                }
            }
        });
    },

    startProcess: function(couchEnvironment) {
        var env_array = couchEnvironment.split(':');
        var port = env_array[0];
        var consumer_key = env_array[1];
        var consumer_secret = env_array[2];
        var token = env_array[3];
        var token_secret = env_array[4];

        CouchDB.port = port;
        CouchDB.accessor = {
            consumerSecret: consumer_secret,
            tokenSecret: token_secret
        };
        CouchDB.message = {
            parameters: {
                oauth_callback: "None",
                oauth_consumer_key: consumer_key,
                oauth_signature_method: "PLAINTEXT",
                oauth_token: token,
                oauth_verifier: "None",
                oauth_version: "1.0"
            }
        };

        Bindwood.couch = new CouchDB('bookmarks');

        try {
            Bindwood.pushBookmarks();
        } catch(e) {
            Bindwood.writeError("Error when calling pushBookmarks: ", e);
        }
        Bindwood.createViews();
        Bindwood.pullBookmarks();
    },

    createViews: function() {
        var views = [{ id: "_design/all_bookmarks",
                       map: "function(doc) { " +
                           "var scheme = doc.uri.split(':',1)[0]; " +
                           "var uri; " +
                           "if (scheme == 'http' || scheme == 'https') {" +
                               "uri = doc.uri.split('/')[2];" +
                               "if (uri.length < 30) {" +
                                   " uri += '/' + " +
                                       "doc.uri.split('/',4)[3].substr(0,30-uri.length) + '...';" +
                               "}" +
                           "} else {" +
                               "uri = scheme + ' URL';" +
                           "}" +
                           "if (!doc.deleted) {" +
                               "emit(doc.title, uri);" +
                           "}" +
                        "}" },
                     { id: "_design/deleted_bookmarks",
                       map: "function(doc) { if (doc.deleted) { emit (doc.title, doc.uri); } }" },
                     { id: "_design/" + Bindwood.currentProfile,
                       map: "function(doc) { var scheme = doc.uri.split(':',1)[0]; var uri; if (scheme == 'http' || scheme == 'https') {uri = doc.uri.split('/')[2];if (uri.length < 30) { uri += '/' + doc.uri.split('/',4)[3].substr(0,30-uri.length) + '...';}} else {uri = scheme + ' URL';}if ((!doc.deleted) && (doc.application_annotations.Firefox.profile == '" + Bindwood.currentProfile + "')) {emit(doc.title, uri);}}" }];
        for (var i = 0; i < views.length; i++) {
            var view_info = views[i];
            var dirty = false;
            try {
                var new_doc;
                var current_doc = Bindwood.couch.open(view_info.id);
                if (current_doc !== null) {
                    var old_map = current_doc.views.display.map;
                    if (old_map != view_info.map) { // Ours is definitive
                        new_doc = current_doc;
                        new_doc.views.display.map = view_info.map;
                        dirty = true;
                    }
                } else {
                    new_doc = {
                        _id: view_info.id,
                        views: {
                            display: {
                                map: view_info.map
                            }
                        }
                    };
                    dirty = true;
                }
                if (dirty) {
                    try {
                        Bindwood.couch.save(new_doc);
                    } catch(e) {
                        Bindwood.writeError("Problem saving view: ", e);
                    }
                }
            } catch(e) {
                // some kind of error fetching the existing design doc
                Bindwood.writeError("Problem checking for view: ", e);
            }
        }
    },

    // Looking up records locally
    annotateItemWithUUID: function(itemId, seed_uuid) {
        var uuid = seed_uuid ? seed_uuid : Bindwood.uuidService.generateUUID().toString();
        Bindwood.annotationService.setItemAnnotation(itemId, Bindwood.annotationKey, uuid, 0, Bindwood.annotationService.EXPIRE_NEVER);
        // Whenever we create a new UUID, stash it and the itemId in
        // our local cache.
        Bindwood.uuidItemIdMap[uuid] = itemId;
        return uuid;
    },

    itemIdForUUID: function(uuid) {
        // First, try to look it up in our local cache, barring that
        // (which shouldn't happen), look it up slowly.
        var itemId = Bindwood.uuidItemIdMap[uuid];

        if (!itemId) {
            var items = Bindwood.annotationService.getItemsWithAnnotation(Bindwood.annotationKey, {});
            var num_items = items.length;
            for (var i = 0; i < items.length; i++) {
                if (Bindwood.annotationService.getItemAnnotation(items[i], Bindwood.annotationKey) == uuid) {
                    Bindwood.uuidItemIdMap[uuid] = itemId = items[i];
                    break;
                }
            }
            if (!itemId) {
                Bindwood.writeMessage("XXX: Still haven't found the right itemId!");
            }
        }
        return itemId;
    },

    uuidForItemId: function(itemId) {
        // Try to look up the uuid, and failing that, assign a new one
        // and return it.
        var uuid;
        try {
            uuid = Bindwood.annotationService.getItemAnnotation(itemId, Bindwood.annotationKey);
        } catch(e) {
            Bindwood.writeError("Couldn't find a UUID for itemId: " + itemId, e);
            uuid = Bindwood.makeLocalChangeOnly(
                function() { return Bindwood.annotateItemWithUUID(itemId, null); } );
        }

        return uuid;
    },

    resolveLocalBookmark: function(bm) {
        // This function is only used to resolve bookmarks from Couch with uuids
        var couch_uuid = bm.application_annotations.Firefox.uuid;
        var itemId = Bindwood.itemIdForUUID(couch_uuid);
        if (itemId) {
            return itemId;
        } else {
            // This bookmark has a uuid, but it's not one of ours.
            // We need to work out whether (a) it's the same as one
            // of ours but with a different uuid (so we need to
            // make the uuids the same), or (b) it's a new one
            // that happens to have been created on a different
            // machine.
            try {
                var uri = Bindwood.ioService.newURI(bm.uri, null, null);
            } catch(e) {
                Bindwood.writeError("Problem creating URI (" + bm.uri + ") for bookmark, skipping: ", e);
                throw e;
            }
            var ids = Bindwood.bookmarksService.getBookmarkIdsForURI(uri, {});
            if (ids.length == 0) {
                return;
            }
            Bindwood.writeMessage("Returning the first bookmark for that URI");
            return ids[0]; // XXX: Just return the first one. Right now, we don't worry too much about duplicates.
        }
    },

    couchReadyObjectForNodeInBookmarksFolder: function(node, bookmarksFolderName) {
        var itemId = node.itemId;

        var title = Bindwood.bookmarksService.getItemTitle(itemId);
        var uuid = Bindwood.uuidForItemId(itemId);
        var folder = bookmarksFolderName;
        var profile = Bindwood.currentProfile;

        var bookmark = {
            title: title,
            application_annotations: {
                Firefox: {
                    uuid: uuid,
                    folder: folder,
                    profile: profile
                }
            }
        };

        switch(Bindwood.bookmarksService.getItemType(itemId)) {
        case Bindwood.bookmarksService.TYPE_BOOKMARK:
            bookmark.record_type = "http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark";
            bookmark.uri = Bindwood.bookmarksService.getBookmarkURI(itemId).spec;
            break;
        case Bindwood.bookmarksService.TYPE_FOLDER:
            bookmark.record_type = "http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder";
            break;
        case Bindwood.bookmarksService.TYPE_SEPARATOR:
            bookmark.record_type = "http://www.freedesktop.org/wiki/Specifications/desktopcouch/separator";
            break;
        default:
            break;
        }

        Bindwood.writeMessage("Prepared bookmark (" + uuid + ") for Couch: " + JSON.stringify(bookmark));
        return bookmark;
    },

    makeLocalChangeOnly: function(func) {
        Bindwood.push = 'DISABLED';
        var results = func();
        Bindwood.push = 'ENABLED';
        return results;
    },

    // Back and forth
    pushBookmarks: function() {
        // Prime the pump, so to speak, by uploading all our local
        // bookmarks to CouchDB (if they're not there already).
        // Create the DB if it doesn't already exist
        try {
            Bindwood.couch.createDb();
        } catch (e) {
            if (e.error == 'file_exists') {
                Bindwood.writeMessage("Database already exists. We're okay.");
            } else {
                Bindwood.writeError("Error when creating database in pushBookmarks: ", e);
            }
        }

        try {
            Bindwood.pushBookmarksFromFolder("toolbarFolder", Bindwood.bookmarksService.toolbarFolder);
        } catch(e) {
            Bindwood.writeError("Error pushing toolbarFolder bookmarks: ", e);
        }

        try {
            Bindwood.pushBookmarksFromFolder("bookmarksMenuFolder", Bindwood.bookmarksService.bookmarksMenuFolder);
        } catch(e) {
            Bindwood.writeError("Error pushing bookmarksMenuFolder bookmarks: ", e);
        }

        try {
            Bindwood.pushBookmarksFromFolder("unfiledBookmarksFolder", Bindwood.bookmarksService.unfiledBookmarksFolder);
        } catch(e) {
            Bindwood.writeError("Error pushing unfiledBookmarksFolder bookmarks: ", e);
        }
    },

    getBookmarksFolder: function(bookmarksFolder) {
        var options = Bindwood.historyService.getNewQueryOptions();
        var query = Bindwood.historyService.getNewQuery();
        query.setFolders([bookmarksFolder], 1);
        var result = Bindwood.historyService.executeQuery(query, options);
        return result.root;
    },

    getBookmarksFromFolder: function(bookmarksFolderName, bookmarksFolder, accum) {
        Bindwood.writeMessage("Fetching bookmarks from: " + bookmarksFolderName);
        var rootNode = Bindwood.getBookmarksFolder(bookmarksFolder);
        rootNode.containerOpen = true;
        for (var i=0; i<rootNode.childCount; i++) {
            var node = rootNode.getChild(i);
            // If node is a Livemark container or a Dynamic container, skip it
            if (PlacesUtils.nodeIsDynamicContainer(node) ||
                PlacesUtils.nodeIsLivemarkContainer(node)) {
                Bindwood.writeMessage(
                    "Skipping Dynamic or Livemark container: " + node.title);
                continue;
            }
            // If node is a folder, descend into it, looking for its contents
            if (node.type == 6) { // RESULT_TYPE_FOLDER
                Bindwood.writeMessage("Descending into folder: " + node.title);
                accum = accum.concat(Bindwood.getBookmarksFromFolder(node.title, node.itemId, accum));
            } else if (node.type == 0 || node.type == 7) { // RESULT_TYPE_URI or RESULT_TYPE_SEPARATOR
                Bindwood.writeMessage("Adding current node: " + node.title);
                accum.push([node.itemId, Bindwood.couchReadyObjectForNodeInBookmarksFolder(node, bookmarksFolderName)]);
            }
        }
        rootNode.containerOpen = false;
        return accum;
    },

    pushBookmarksFromFolder: function(bookmarksFolderName, bookmarksFolder) {
        Bindwood.writeMessage("Pushing bookmarks from " + bookmarksFolderName);
        var bookmarkData = Bindwood.getBookmarksFromFolder(bookmarksFolderName, bookmarksFolder, new Array());
        for (var i = 0; i < bookmarkData.length; i++) {
            // find this bookmark in CouchDB
            var itemId = bookmarkData[i][0];
            var bookmark = bookmarkData[i][1];
            var uuid = bookmark.application_annotations.Firefox.uuid;
            Bindwood.writeMessage("Bookmark so far: " + JSON.stringify(bookmark));

            // Even though this happens in annotateItemWithUUID as well,
            //  we're priming the cache here for first run.
            Bindwood.uuidItemIdMap[uuid] = itemId;

            try {
                var results = Bindwood.couch.query(function(doc) {
                                           if (doc.application_annotations &&
                                               doc.application_annotations.Firefox &&
                                               doc.application_annotations.Firefox.uuid) {
                                               emit(doc.application_annotations.Firefox.uuid, doc);
                                           }
                                       }, null, {
                                           startkey: uuid, endkey: uuid
                                       });
            } catch(e) {
                Bindwood.writeError("Error querying couch: ", e);
            }

            if (results.rows.length === 0) {
                // this bookmark is not in CouchDB, so write it
                try {
                    Bindwood.couch.save(bookmark);
                    Bindwood.writeMessage("Saved bookmark (" + uuid + ") to Couch.");
                } catch(e) {
                    Bindwood.writeError("Problem saving bookmark to CouchDB; bookmark is " + JSON.stringify(bookmark) + ": ", e);
                }
            } else {
                Bindwood.writeMessage("This bookmark (" + uuid + ") is already in Couch, skipping");
                // bookmark is already in CouchDB, so do nothing
            }
        }
    },

    pullBookmarks: function() {
        // Fetch all bookmark documents from the database
        // The query function is evaluated by Couch, which doesn't know
        // what Bindwood.RECORD_TYPE is, so we string-encode it first to
        // include the literal value
        Bindwood.writeMessage("Fetching bookmarks for " + Bindwood.currentProfile);
        var rows = {rows: []};
        try {
            rows = Bindwood.couch.query(
                "function (doc) {" +
                    " if (doc.record_type == \"http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark\" &&" +
                    " doc.application_annotations &&" +
                    " doc.application_annotations.Firefox &&" +
                    " doc.application_annotations.Firefox.profile &&" +
                    " doc.application_annotations.Firefox.profile == \"" + Bindwood.currentProfile + "\") {" +
                    " emit(doc._id,doc);}}");
        } catch(e) {
            Bindwood.writeError("Problem fetching all bookmarks from Couch: ", e);
        }
        Bindwood.writeMessage("We got back " + rows.rows.length + " rows");
        for (var i = 0; i < rows.rows.length; i++) {
            var recordid = rows.rows[i].id;
            Bindwood.writeMessage("Pulling record: " + recordid);
            var bm = rows.rows[i].value;

            // First, check to see if the bookmark we've pulled down is flagged as deleted.
            // If so, we should make sure any local copy we have of this bookmark has also been deleted.
            if (bm.deleted) {
                Bindwood.writeMessage("Bookmark in Couch marked as deleted; attempting to delete local copy.");
                Bindwood.deleteLocalBookmark(bm);
            } else if (bm.application_annotations &&
                       bm.application_annotations.Firefox &&
                       bm.application_annotations.Firefox.uuid) {
                Bindwood.writeMessage("Bookmark in Couch has UUID, merging local copy.");
                Bindwood.mergeLocalBookmark(bm, recordid);
            } else {
                Bindwood.writeMessage("Bookmark in Couch has no UUID, creating new local copy, annotating, and pushing back.");
                Bindwood.addLocalBookmark(bm, recordid, null);
            }
        }
        // reschedule ourself
        Bindwood.writeMessage("Successful run, rescheduling ourself");
        setTimeout(Bindwood.pullBookmarks, 30000);
    },

    deleteLocalBookmark: function(bm) {
        // If we can't resolve the itemId, even by looking up URI, assume it's already gone.
        var itemId = Bindwood.resolveLocalBookmark(bm);
        if (itemId) {
            Bindwood.makeLocalChangeOnly(
                function() { return Bindwood.bookmarksService.removeItem(itemId); });
        }
    },

    mergeLocalBookmark: function(bm, recordid) {
        // this bookmark has a uuid, so check its values haven't changed
        // find the bookmark with this uuid
        var itemId = Bindwood.resolveLocalBookmark(bm);
        var couch_uuid = bm.application_annotations.Firefox.uuid;

        if (itemId) {
            // Found one local bookmark. Replace its uuid to
            // be the one from Couch.
            var old_uuid = Bindwood.uuidForItemId(itemId);
            if (old_uuid != couch_uuid) {
                delete Bindwood.uuidItemIdMap[old_uuid];
                Bindwood.makeLocalChangeOnly(
                    function() { return Bindwood.annotateItemWithUUID(itemId, couch_uuid); } );
            }

            // XXX: I'm still not positive about this. While I'm far more sure that we're seeding
            // Couch properly from the beginning and properly making new bookmarks, I still wonder
            // if we'll see cases where Couch has null for title and uri, and how best to manage
            // them. I'm thinking timestamps. - urbanape
            var title = Bindwood.bookmarksService.getItemTitle(itemId);
            if (title != bm.title) {
                Bindwood.writeMessage("Resetting local title to title from Couch");
                Bindwood.makeLocalChangeOnly(
                    function() { return Bindwood.bookmarksService.setItemTitle(itemId, bm.title); });
            }

            var metadata = Bindwood.bookmarksService.getBookmarkURI(itemId);
            if (metadata.spec != bm.uri) {
                Bindwood.writeMessage("The URI from Couch (" + bm.uri + ") is different from local (" + metadata.spec + ")");
                try {
                    var new_uri = Bindwood.ioService.newURI(bm.uri, null, null);
                    Bindwood.writeMessage("Creating a new URI for our local bookmark");
                    Bindwood.makeLocalChangeOnly(
                        function() { return Bindwood.bookmarksService.changeBookmarkURI(itemId, new_uri); });
                } catch(e) {
                    Bindwood.writeError("Problem creating a new URI for bookmark: ", e);
                }
            }
        } else {
            /// No local bookmarks
            Bindwood.writeMessage("No local bookmark found, must be a new entry in Couch. Creating locally.");
            Bindwood.addLocalBookmark(bm, recordid, couch_uuid);
        }
    },

    guaranteeDesktopCouchFolder: function() {
        // If we're syncing from CouchDB and we have new bookmarks,
        // until we have proper replication of hierarchy and folders,
        // we should put all "new" bookmarks in a collected, sane
        // place. Unfiled bookmarks are hard to get to (only available
        // through "Organize bookmarks"), so we'll put them in a
        // folder in the Boomarks Menu called "Desktop Couch",
        // creating that folder first, if necessary.
        var bookmarksFolder = Bindwood.bookmarksService.bookmarksMenuFolder;
        var rootNode = Bindwood.getBookmarksFolder(bookmarksFolder);
        rootNode.containerOpen = true;
        for (var i=0; i<rootNode.childCount; i++) {
            var node = rootNode.getChild(i);
            if (node.title == 'Desktop Couch') {
                rootNode.containerOpen = false;
                return node.itemId;
            }
        }
        rootNode.containerOpen = false;

        var folderId = Bindwood.bookmarksService.createFolder(
            bookmarksFolder, 'Desktop Couch', -1);
        return folderId;
    },

    addLocalBookmark: function(bm, recordid, uuid) {
        // If uuid is present, we only need create the bookmark locally,
        // as it already exists (with uuid) in Couch.
        //
        // If uuid is null, then after we create it locally,
        // we want to annotate it and push back to Couch.
        //
        // By default, store new bookmarks in the unfiledBookmarksFolder
        var folder;
        if (bm.application_annotations &&
            bm.application_annotations.Firefox &&
            bm.application_annotations.Firefox.folder) {
            switch (bm.application_annotations.Firefox.folder) {
            case "toolbarFolder":
                folder = Bindwood.bookmarksService.toolbarFolder;
                break;
            case "bookmarksMenuFolder":
                folder = Bindwood.bookmarksService.bookmarksMenuFolder;
                break;
            default:
                folder = Bindwood.guaranteeDesktopCouchFolder();
                break;
            }
        }

        try {
            var new_uri = Bindwood.ioService.newURI(bm.uri, null, null);
        } catch(e) {
            Bindwood.writeError("Problem creating a new URI for bookmark: ", e);
        }

        var itemId = Bindwood.makeLocalChangeOnly(
            function() { return Bindwood.bookmarksService.insertBookmark(
                             folder, new_uri, -1, bm.title); });

        if (uuid) { // We were provided a uuid, so no need to send back to Couch, just annotate locally
            Bindwood.makeLocalChangeOnly(
                function() { return Bindwood.annotateItemWithUUID(itemId, uuid); } );
        } else {  // This is an unadorned bookmark from Couch. Save it to Couch once we establish its uuid
            var new_uuid = Bindwood.uuidForItemId(itemId);
            var doc = Bindwood.couch.open(recordid);
            if (!doc.application_annotations) {
                doc.application_annotations = {};
            }
            if (!doc.application_annotations.Firefox) {
                doc.application_annotations.Firefox = {};
            }
            doc.application_annotations.Firefox.uuid = new_uuid;
            try {
                Bindwood.couch.save(doc);
                Bindwood.writeMessage("Saved the newly annotated doc back to Couch");
            } catch(e) {
                Bindwood.writeError("Problem writing record for new bookmark: ",e);
            }
        }
    },

    findDocumentByUUID: function(uuid) {
        Bindwood.writeMessage("Looking up a document in Couch by uuid: " + uuid);
        var results = Bindwood.couch.query(function(doc) {
        if (doc.application_annotations &&
                doc.application_annotations.Firefox &&
                doc.application_annotations.Firefox.uuid) {
                emit(doc.application_annotations.Firefox.uuid, doc);
            }
        }, null, {
            startkey: uuid, endkey: uuid
        });
        if (results.rows.length === 0) {
            Bindwood.writeMessage("Problem finding document");
            return;
        }
        Bindwood.writeMessage("Found: " + JSON.stringify(results));
        return results;
    },

    updateDocAndSave: function(uuid, attribute, value, callback) {
        Bindwood.writeMessage("Updating a document (" + uuid + ") setting (" + attribute + ") to (" + value + ")");
        // Some attributes that we track should remain inside the application_annotations object
        var attrMap = {
             title: false,
             uri: false,
             deleted: false,
             uuid: true,
             folder: true,
             favicon: true,
             profle: true };
        var results = Bindwood.findDocumentByUUID(uuid);
        if (!results) {
            throw {error: "Could not find document in Couch"};
        }
        var doc = Bindwood.couch.open(results.rows[0].id);
        if (attrMap[attribute.toString()]) { // belongs on annotations
            if (!doc.application_annotations) {
                doc.application_annotations = {};
            }
            if (!doc.application_annotations.Firefox) {
                doc.application_annotations.Firefox = {};
            }
            doc.application_annotations.Firefox[attribute.toString()] = value.toString();
        } else {
            doc[attribute.toString()] = value.toString();
        }
        try {
            var result = Bindwood.couch.save(doc);
            Bindwood.writeMessage("Successfully save document (" + uuid + ") back to Couch.");
        } catch(e) {
            Bindwood.writeError("Problem saving document to Couch", e);
            throw e;
        }

        if (callback) {
            Bindwood.writeMessage("We've got a callback to run: " + callback.toString());
            callback();
        }

        return result;
    },

    itemWeCareAbout: function(itemId) {
        // Traverse from the itemId up its parent chain. If at any
        // level the parent is a livemark container or a dynamic
        // container, return false, otherwise, return true.
        var root = 0;
        var parent;
        while (parent != root) {
            parent = Bindwood.bookmarksService.getFolderIdForItem(itemId);
            if (parent != root &&
                Bindwood.annotationService.itemHasAnnotation(
                    parent, 'livemark/feedURI')) {
                return false;
            }
            itemId = parent;
        }
        return true;
    },

    Observer: {
        // An nsINavBookmarkObserver
        onItemAdded: function(aItemId, aFolder, aIndex) {
            // A bookmark has been added, so we create a blank entry
            // in Couch with our local itemId attached.
            if (!Bindwood.itemWeCareAbout(aItemId)) {
                Bindwood.writeMessage("Ignoring this add event");
                return;
            }

            Bindwood.writeMessage("A new bookmark was created. Its id is: " + aItemId +
                                  " at location: " + aIndex +
                                  " in folder: " + aFolder );
            netscape.security.PrivilegeManager.enablePrivilege("UniversalBrowserRead UniversalBrowserWrite");

            var uuid = Bindwood.uuidForItemId(aItemId);
            Bindwood.writeMessage("Determined uuid for new bookmark: " + uuid);

            var folder;
            switch(aFolder) {
                case Bindwood.bookmarksService.toolbarFolder:
                    folder = "toolbarFolder";
                    break;
                case Bindwood.bookmarksService.bookmarksMenuFolder:
                    folder = "bookmarksMenuFolder";
                    break;
                case Bindwood.bookmarksService.unfiledBookmarksFolder:
                    folder = "unfiledBookmarksFolder";
                    break;
                default:
                    folder = Bindwood.bookmarksService.getItemTitle(aFolder);
                    break;
            }
            var uri;
            var title;

            Bindwood.writeMessage("Going to look up bookmark URI");
            try {
                uri = Bindwood.bookmarksService.getBookmarkURI(aItemId).spec;
            } catch(e) {
                Bindwood.writeError("Problem with something: ", error);
                uri = '';
            }

            Bindwood.writeMessage("Going to look up bookmark Title");
            try {
                title = Bindwood.bookmarksService.getItemTitle(aItemId);
            } catch(e) {
                Bindwood.writeError("Problem with something: ", error);
                title = '';
            }

            var doc = {
                record_type: "http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark",
                uri: uri,
                title: title,
                application_annotations: {
                    Firefox: {
                        uuid: uuid,
                        folder: folder,
                        profile: Bindwood.currentProfile
                    }
                }
            };

            Bindwood.writeMessage("Created a minimal record document with our uuid: " + JSON.stringify(doc));

            switch (Bindwood.push) {
            case 'DISABLED':
                Bindwood.writeMessage("Added, but not saving to Couch.");
                break;
            case 'ENABLED':
                try {
                    var result = Bindwood.couch.save(doc);
                    Bindwood.writeMessage("Saved new, bare record to Couch.");
                } catch(e) {
                    Bindwood.writeError("Problem saving new bookmark to Couch: ", e);
                }
                break;
            default:
                break;
            }
        },
        onBeforeItemRemoved: function(aItemId) {
            // A bookmark has been removed. This is called before it's
            // been removed locally, though we're passed the itemId,
            // which we use to delete from Couch.
            if (!Bindwood.itemWeCareAbout(aItemId)) {
                Bindwood.writeMessage("Ignoring this before remove event");
                return;
            }

            Bindwood.writeMessage("Record " + aItemId + " is about to be removed locally.");
            var uuid = Bindwood.uuidForItemId(aItemId);

            switch (Bindwood.push) {
            case 'DISABLED':
                delete Bindwood.uuidItemIdMap[uuid];
                Bindwood.writeMessage("Deleted from local uuid map, but not saving back to Couch.");
                break;
            case 'ENABLED':
                netscape.security.PrivilegeManager.enablePrivilege("UniversalBrowserRead UniversalBrowserWrite");

                try {
                    // Also remove from our local cache and remove
                    // annotation from service.
                    var result = Bindwood.updateDocAndSave(
                        uuid, 'deleted', true,
                        function() {
                            delete Bindwood.uuidItemIdMap[uuid];
                            Bindwood.writeMessage("Deleted local reference in the uuid-itemId mapping."); });
                    Bindwood.writeMessage("Saved document back to Couch with deleted flag set.");
                } catch(e) {
                    Bindwood.writeError("Problem pushing deleted record to Couch: ", e);
                }
                break;
            default:
                break;
            }
        },
        onItemRemoved: function(aItemId, aFolder, aIndex) {
            // This only happens locally, so there's never a need to push
            if (!Bindwood.itemWeCareAbout(aItemId)) {
                Bindwood.writeMessage("Ignoring this remove event");
                return;
            }

            Bindwood.makeLocalChangeOnly(
                function() { return Bindwood.annotationService.removeItemAnnotation(aItemId, Bindwood.annotationKey); });
            Bindwood.writeMessage("Removed annotations from bookmark identified by: " + aItemId);
        },
        onItemChanged: function(aBookmarkId, aProperty, aIsAnnotationProperty, aValue) {
            // A property of a bookmark has changed. On multiple
            // property updates, this will be called multiple times,
            // once per property (i.e., for title and URI)
            if (!Bindwood.itemWeCareAbout(aItemId)) {
                Bindwood.writeMessage("Ignoring this change event");
                return;
            }

            if (!aIsAnnotationProperty) { // We only care if these are bookmark properties (for right now)
                Bindwood.writeMessage("A property (" + aProperty + ") on bookmark id: " + aBookmarkId + " has been set to: " + aValue);
                var uuid = Bindwood.uuidForItemId(aBookmarkId);

                switch (Bindwood.push) {
                case 'DISABLED':
                    Bindwood.writeMessage("Updated, but not saving back to Couch.");
                    break;
                case 'ENABLED':
                    Bindwood.writeMessage("We will push this change back to Couch.");
                    netscape.security.PrivilegeManager.enablePrivilege("UniversalBrowserRead UniversalBrowserWrite");
                    try {
                        var result = Bindwood.updateDocAndSave(
                            uuid, aProperty.toString(), aValue.toString(),
                            function() { Bindwood.writeMessage("Saved the document back to Couch"); });
                    } catch(e) {
                        Bindwood.writeError("Problem saving updated bookmark to Couch: ", e);
                    }
                    break;
                default:
                    break;
                }
            }
        },

        // Currently unhandled
        onBeginUpdateBatch: function() {},
        onEndUpdateBatch: function() {},
        onItemMoved: function(aItemId, aOldParent, aOldIndex, aNewParent, aNewIndex) {},
        onItemVisited: function(aBookmarkId, aVisitID, time) {},

        // Query Interface
        QueryInterface: function(iid) {
            if (iid.equals(Ci.nsINavBookmarkObserver) ||
                iid.equals(Ci.nsINavBookmarkObserver_MOZILLA_1_9_1_ADDITIONS) ||
                iid.equals(Ci.nsISupports)) {
                return this;
            }
            throw Cr.NS_ERROR_NO_INTERFACE;
        }
    }
};
