/* specter_PGSQL.c
 *
 * output plugin for logging data into a PostgreSQL database
 * 
 * (C) 2004,2005 by Michal Kwiatkowski <ruby@joker.linuxstuff.pl>
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <specter/specter.h>
#include <conffile/conffile.h>
#include <libpq-fe.h>
#include "sql.h"


/* our configuration directives */
static config_entry_t my_config[] = {
	{ .key = "db", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_MANDATORY },
	{ .key = "host", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "user", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "pass", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
	{ .key = "table", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_MANDATORY },
	{ .key = "buffsize", .type = CONFIG_TYPE_MEM, .options = CONFIG_OPT_NONE, .u = { .value = 0 } },
	{ .key = "ssl_enable", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE, .u = { .value = 0 } },
	{ .key = "port", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE },
};

struct my_data {
	PGconn *dbh;
	struct sql_field *f;
	char *buff_start;
	char *buff_cur;
	size_t length;
};


static void *specter_pgsql_init(config_entry_t *ce)
{
	struct my_data *data;
	PGresult *result;
	char *initbuff, **columns;
	int initlen;
	int ctr, col_num;

	if ((data = malloc(sizeof(struct my_data))) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		return NULL;
	}

	memset(data, 0x0, sizeof(struct my_data));

	/* 35: "host= dbname= user= password= port=" */
	initlen = 35 + 1 + strlen(GET_CE(ce,0)->u.string)
		+ strlen(GET_CE(ce,1)->u.string)
		+ strlen(GET_CE(ce,2)->u.string)
		+ strlen(GET_CE(ce,3)->u.string)
		+ (GET_CE(ce,6)->u.value) ? 13 : 0;
	if ((initbuff = malloc(initlen)) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		goto out_free_data;
	}

	sprintf(initbuff, "dbname=%s", GET_CE(ce,0)->u.string);
	if (GET_CE(ce,1)->u.string[0] != '\0') {
		strcat(initbuff, " host=");
		strcat(initbuff, GET_CE(ce,1)->u.string);
	}
	if (GET_CE(ce,2)->u.string[0] != '\0') {
		strcat(initbuff, " user=");
		strcat(initbuff, GET_CE(ce,2)->u.string);
	}
	if (GET_CE(ce,3)->u.string[0] != '\0') {
		strcat(initbuff, " password=");
		strcat(initbuff, GET_CE(ce,3)->u.string);
	}
	if (GET_CE(ce,6)->u.value) {
		strcat(initbuff, " requiressl=1");
	}
	if (GET_CE(ce,7)->u.string[0] != '\0') {
		/*
		 * PostgreSQL documentation defines port as:
		 *   Port number to connect to at the server host, or socket
		 *   file name extension for Unix-domain connections.
		 * thus using CONFIG_TYPE_STRING, not CONFIG_TYPE_INT.
		 */
		strcat(initbuff, " port=");
		strcat(initbuff, GET_CE(ce,7)->u.string);
	}

	data->dbh = PQconnectdb(initbuff);
	if (PQstatus(data->dbh) != CONNECTION_OK) {
		specter_log(SPECTER_FATAL, "Couldn't connect to database: %s.\n",
				PQerrorMessage(data->dbh));
		goto out_free_initbuff;
	}

	/* 123: "SELECT ... ORDER BY a.attnum" */
	initlen = 123 + 1 + strlen(GET_CE(ce,4)->u.string);
	if ((initbuff = realloc(initbuff, initlen)) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		goto out_free_initbuff;
	}
	sprintf(initbuff, "SELECT  a.attname FROM pg_class c, pg_attribute a "
			"WHERE c.relname ='%s' AND a.attnum>0 AND a.attrelid=c.oid "
			"ORDER BY a.attnum", GET_CE(ce,4)->u.string);

	if ((result = PQexec(data->dbh, initbuff)) == NULL
			|| PQresultStatus(result) != PGRES_TUPLES_OK) {
		specter_log(SPECTER_FATAL, "Couldn't get pgsql columns: %s.\n",
				PQerrorMessage(data->dbh));
		goto out_free_initbuff;
	}

	col_num = PQntuples(result);
	if ((columns = malloc((col_num + 1) * sizeof(char *))) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		goto out_free_result;
	}

	for (ctr = 0; ctr < col_num; ctr++)
		columns[ctr] = PQgetvalue(result, ctr, 0);
	columns[ctr] = NULL;

	data->length = GET_CE(ce,5)->u.value;
	data->buff_cur = alloc_sql_insert(columns, GET_CE(ce,4)->u.string,
			&data->buff_start, &data->length, &data->f);
	/* alloc_sql_insert already printed error message */
	if (data->buff_cur == NULL)
		goto out_free_columns;

	/* all done */
	free(columns);
	PQclear(result);
	free(initbuff);
	return data;

out_free_columns:
	free(columns);

out_free_result:
	PQclear(result);

out_free_initbuff:
	PQfinish(data->dbh);
	free(initbuff);

out_free_data:
	free(data);

	return NULL;
}


static PGconn *global_pgsql_dbh;
static size_t specter_pgsql_escape(char *dest, const char *src, size_t n)
{
	unsigned long length = strlen(src);

	if (n < length*2+1) /* will cause exit in fill_sql_insert() */
		return length*2+1;

	*dest++ = '\'';
	length = PQescapeString(dest, src, length);
	*(dest+length) = '\'';

	return length+2;
}


static int specter_pgsql_output(config_entry_t *ce, void *data)
{
	struct my_data *md = data;
	char *end;
	PGresult *result;

	global_pgsql_dbh = md->dbh;
	if ((end = fill_sql_insert(md->f, md->buff_cur,
				md->length - (md->buff_cur - md->buff_start),
				&specter_pgsql_escape)) == NULL)
		return -1;
	global_pgsql_dbh = NULL;

	specter_log(SPECTER_DEBUG, "Calling PostgreSQL: #%s#\n", md->buff_start);

	if ((result = PQexec(md->dbh, md->buff_start)) == NULL
			|| PQresultStatus(result) != PGRES_COMMAND_OK) {
		specter_log(SPECTER_ERROR, "Couldn't insert data into database: %s.\n",
				PQerrorMessage(md->dbh));
		return -1;
	}

	return 0;
}


static void specter_pgsql_fini(config_entry_t *ce, void *data)
{
	struct my_data *md = data;

	PQfinish(md->dbh);
	free_sql_insert(md->buff_start, md->f);
	free(data);
}


static specter_output_t pgsql_op = {
	.name = "pgsql",
	.ce_base = my_config,
	.ce_num = 8,
	.init = &specter_pgsql_init,
	.fini = &specter_pgsql_fini,
	.output = &specter_pgsql_output
};

void _init(void) 
{
	if (register_output(&pgsql_op, 0) == -1) {
		specter_log(SPECTER_FATAL, "Can't register.\n");
		exit(EXIT_FAILURE);
	}
}

