/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
using Dee;
using DBus;
using Zeitgeist;
using Zeitgeist.Timestamp;
using Unity.Place;
using Config;
using Gee;
using GMenu;

namespace Unity.ApplicationsPlace {

  const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";

  public class Daemon : GLib.Object, Unity.Place.Activation
  {
    private Zeitgeist.Log log;
    private Zeitgeist.Index zg_index;
    private Unity.Package.Searcher pkgsearcher;
    private Unity.Package.Searcher appsearcher;

    private Unity.Place.Controller control;
    private Unity.Place.EntryInfo applications;

    /* For each section we have a set filtering query we use to restrict
     * Xapian queries to that section */
    private Gee.List<string> section_queries;
    private Gee.List<Set<string>> section_categories;

    private Gee.List<string> image_extensions;
    private HashTable<string,Icon> file_icon_cache;
    
    /* We remember the previous search so we can figure out if we should do
     * incremental filtering of the result models */
    private Search? previous_search;
    private Search? previous_global_search;
    
    /* To make sure we don't fire of unnecessary queries if the active section
     * is in fact not changed */
    private uint previous_active_section;
    
    private Dee.Index entry_results_by_group;
    private Dee.Index global_results_by_group;
    
    private PtrArray zg_templates;
    
    /* Gnome menu structure - also used to check whether apps are installed */
    private uint app_menu_changed_reindex_timeout = 0;
    private GMenu.Tree app_menu = null;

    construct
    {
      var sections_model = new Dee.SharedModel(
                                 "com.canonical.Unity.ApplicationsPlace.SectionsModel",
                                 2, typeof (string), typeof (string));

      var groups_model = new Dee.SharedModel(
                                 "com.canonical.Unity.ApplicationsPlace.GroupsModel",
                                 3, typeof (string), typeof (string),
                                 typeof (string));

      var global_groups_model = new Dee.SharedModel(
                                 "com.canonical.Unity.ApplicationsPlace.GlobalGroupsModel",
                                 3, typeof (string), typeof (string),
                                 typeof (string));

      var results_model = new Dee.SharedModel(
                                 "com.canonical.Unity.ApplicationsPlace.ResultsModel",
                                 6, typeof (string), typeof (string),
                                 typeof (uint), typeof (string),
                                 typeof (string), typeof (string));

      var global_results_model = new Dee.SharedModel(
                                 "com.canonical.Unity.ApplicationsPlace.GlobalResultsModel",
                                 6, typeof (string), typeof (string),
                                 typeof (uint), typeof (string),
                                 typeof (string), typeof (string));

      section_queries = new Gee.ArrayList<string> ();
      section_categories = new Gee.ArrayList<Set<string>> ();
      populate_section_queries();
      populate_section_categories();
      populate_zg_templates ();

      applications = new EntryInfo ("/com/canonical/unity/applicationsplace/applications");
      applications.sections_model = sections_model;
      applications.entry_renderer_info.groups_model = groups_model;
      applications.entry_renderer_info.results_model = results_model;
      applications.global_renderer_info.groups_model = global_groups_model;
      applications.global_renderer_info.results_model = global_results_model;

      populate_sections ();
      populate_groups ();
      populate_global_groups ();

      applications.icon = @"$(Config.PREFIX)/share/unity/applications.png";

      log = new Zeitgeist.Log();
      zg_index = new Zeitgeist.Index();
      pkgsearcher = new Unity.Package.Searcher ();
      
      /* Image file extensions in order of popularity */
      image_extensions = new Gee.ArrayList<string> ();
      image_extensions.add ("png");
      image_extensions.add ("xpm");
      image_extensions.add ("svg");
      image_extensions.add ("tiff");
      image_extensions.add ("ico");
      image_extensions.add ("tif");
      image_extensions.add ("jpg");
      
      previous_search = null;
      previous_global_search = null;
      previous_active_section = Section.LAST_SECTION; /* Must be an invalid section! */
      
      /* Create indexes on the result models */
      var analyzer = new Dee.Analyzer.for_uint_column (ResultsColumn.GROUP_ID);
      entry_results_by_group = new Dee.HashIndex(results_model, analyzer);
      global_results_by_group = new Dee.HashIndex(global_results_model, analyzer);
      
      build_app_menu_index ();
      
      file_icon_cache = new HashTable<string,Icon>(str_hash, str_equal);
      
      /* Listen for section changes */
      applications.notify["active-section"].connect (
        (obj, pspec) => {
          if (previous_active_section == applications.active_section)
            return;
          
          var _results_model = applications.entry_renderer_info.results_model;
          var _groups_model = applications.entry_renderer_info.groups_model;
          var search = applications.active_search;
          search_async.begin (search,
                              (Section)applications.active_section,
                              _results_model, _groups_model,
                              false,
                              entry_results_by_group);
          previous_search = search;
          previous_active_section = applications.active_section;
        }
      );

      /* Listen for changes to the place entry search */
      applications.notify["active-search"].connect (
        (obj, pspec) => {
          var search = applications.active_search;
          if (!Utils.search_has_really_changed (previous_search, search))
            return;
          var _results_model = applications.entry_renderer_info.results_model;
          var _groups_model = applications.entry_renderer_info.groups_model;          
          search_async.begin (search,
                              (Section)applications.active_section,
                              _results_model, _groups_model,
                              check_is_filter_search (search, previous_search),
                              entry_results_by_group);
          previous_search = search;
        }
      );

      /* Listen for changes to the global search. In global search we don't
       * search for Most Used apps with Zeitgeist because we don't have
       * meaningful grouping */
      applications.notify["active-global-search"].connect (
        (obj, pspec) => {
          var search = applications.active_global_search;
          if (!Utils.search_has_really_changed (previous_search, search))
            return;
          var _results_model = applications.global_renderer_info.results_model;
          var _groups_model = applications.global_renderer_info.groups_model;
          var _is_filter_search = check_is_filter_search (search,
                                                          previous_global_search);
          
          if (!_is_filter_search)
            {
              _results_model.clear ();
            }
           
          update_pkg_search (search, Section.ALL_APPLICATIONS,
                             _results_model, _is_filter_search,
                             global_results_by_group);
                             
          previous_global_search = search;
        }
      );

      /* We should not do anything with the results model
       * until we receieve the 'ready' signal */
      //results_model.ready.connect (
      //  (model) => { update_entry_results_model.begin(); }
      //);

      /* Listen for changes in the installed applications */
      AppInfoManager.get_instance().changed.connect (on_appinfo_changed);

      sections_model.connect ();
      groups_model.connect ();
      global_groups_model.connect ();
      results_model.connect ();
      global_results_model.connect ();

      /* The last thing we do is export the controller. Once that is up,
       * clients will expect the SharedModels to work */
      control = new Unity.Place.Controller ("/com/canonical/unity/applicationsplace");
      control.add_entry (applications);
      control.activation = this;
      try {
        control.export ();
      } catch (DBus.Error error) {
        critical ("Failed to export DBus service for '%s': %s",
                  control.dbus_path, error.message);
      }

    }

    private void populate_sections ()
    {
      var sections = applications.sections_model;

      if (sections.get_n_rows() != 0)
        {
          critical ("The sections model should be empty before initial population");
          sections.clear ();
        }

      sections.append (SectionsColumn.DISPLAY_NAME, _("All Applications"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Accessories"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Games"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Internet"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Media"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("Office"),
                       SectionsColumn.ICON_HINT, "", -1);
      sections.append (SectionsColumn.DISPLAY_NAME, _("System"),
                       SectionsColumn.ICON_HINT, "", -1);
    }

    private void populate_groups ()
    {
      var groups = applications.entry_renderer_info.groups_model;

      if (groups.get_n_rows() != 0)
        {
          critical ("The groups model should be empty before initial population");
          groups.clear ();
        }

      groups.append (GroupsColumn.RENDERER, "UnityShowcaseRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Most Used"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-mostused.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityDefaultRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Installed"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-installed.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityDefaultRenderer",
                     GroupsColumn.DISPLAY_NAME, _("Available"),
                     GroupsColumn.ICON_HINT, ICON_PATH + "group-available.svg", -1);
      groups.append (GroupsColumn.RENDERER, "UnityEmptySearchRenderer",
                     GroupsColumn.DISPLAY_NAME, "No search results", // No i18n, should never be rendered
                     GroupsColumn.ICON_HINT, "", -1);
      groups.append (GroupsColumn.RENDERER, "UnityEmptySectionRenderer",
                     GroupsColumn.DISPLAY_NAME, "Empty section", // No i18n, should never be rendered
                     GroupsColumn.ICON_HINT, "", -1);
      
      /* Always expand the Installed group */
      applications.entry_renderer_info.set_hint ("ExpandedGroups",
                                                 @"$((uint)Group.INSTALLED)");
    }
    
    private void populate_global_groups ()
    {
      var groups = applications.global_renderer_info.groups_model;

      if (groups.get_n_rows() != 0)
        {
          critical ("The global groups model should be empty " +
                    "before initial population");
          groups.clear ();
        }
      
      /* We only have a single dummy group for global search */
      groups.append (GroupsColumn.RENDERER, "UnityDefaultRenderer",
                     GroupsColumn.DISPLAY_NAME, "",
                     GroupsColumn.ICON_HINT, ICON_PATH + "apps.svg", -1);
    }

    private void populate_section_queries ()
    {
      /* XDG category names. Not for translation. */
      /* We need the hack for ALL_APPLICATIONS below because Xapian doesn't
       * like '' or '*' queries */
      section_queries.add ("NOT category:XYZ"); //ALL_APPLICATIONS
      section_queries.add ("category:Utility"); //ACCESSORIES
      section_queries.add ("category:Game"); //GAMES
      section_queries.add ("category:Network"); //INTERNET
      section_queries.add ("(category:AudioVideo OR category:Graphics)"); //MEDIA
      section_queries.add ("category:Office"); //OFFICE
      section_queries.add ("(category:System OR category:Settings)"); //SYSTEM
    }
    
    private void populate_section_categories ()
    {
      /* XDG category names. Not for translation */
      //ALL_APPLICATIONS
      Set<string> cat = new TreeSet<string>();
      cat.add ("AudioVideo");
      cat.add ("Development");
      cat.add ("Education");
      cat.add ("Game");
      cat.add ("Graphics");
      cat.add ("Network");
      cat.add ("Office");
      cat.add ("Settings");
      cat.add ("System");
      cat.add ("Utility");
      section_categories.add (cat);
      
      //ACCESSORIES
      cat = new HashSet<string>();
      cat.add ("Utility");
      section_categories.add (cat);
      
      //GAMES
      cat = new HashSet<string>();
      cat.add ("Game");
      section_categories.add (cat);
      
      //INTERNET
      cat = new HashSet<string>();
      cat.add ("Network");
      section_categories.add (cat);
      
      //MEDIA
      cat = new HashSet<string>();
      cat.add ("AudioVideo");
      cat.add ("Graphics");
      section_categories.add (cat);
      
      //OFFICE
      cat = new HashSet<string>();
      cat.add ("Office");
      section_categories.add (cat);
      
      //SYSTEM
      cat = new HashSet<string>();
      cat.add ("Settings");
      cat.add ("System");
      section_categories.add (cat);
    }

    /* Load xdg menu info and build a Xapian index over it.
     * Do throttled re-index if the menu changes */
    private bool build_app_menu_index ()
    {            
      if (app_menu == null)
        {
          debug ("Building initial application menu");
        
          /* We need INCLUDE_NODISPLAY to employ proper de-duping between
           * the Installed and Availabale groups. If a NoDisplay app is installed,
           * eg. Evince, it wont otherwise be in the menu index, only in the
           * S-C index - thus show up in the Available group */
          app_menu = GMenu.Tree.lookup ("unity-place-applications.menu",
                                            GMenu.TreeFlags.INCLUDE_NODISPLAY);          
          
          app_menu.add_monitor ((menu) => {
            /* Reschedule the timeout if we already have one. The menu tree triggers
             * many change events during app installation. This way we wait the full
             * delay *after* the last change event has triggered */
            if (app_menu_changed_reindex_timeout != 0)
              Source.remove (app_menu_changed_reindex_timeout);
            
            app_menu_changed_reindex_timeout =
                                  Timeout.add_seconds (5, build_app_menu_index);
          });
        }
      
      debug ("Indexing application menu");
      appsearcher = new Unity.Package.Searcher.for_menu (app_menu);
      app_menu_changed_reindex_timeout = 0;
      
      return false;
    }

    private void populate_zg_templates ()
    {
      /* Create a template that activation of applications */
      zg_templates = new PtrArray.sized(1);
      var ev = new Zeitgeist.Event.full (ZG_ACCESS_EVENT, ZG_USER_ACTIVITY, "",
                             new Subject.full ("application://*",
                                               "", //NFO_SOFTWARE,
                                               "",
                                               "", "", "", ""));
      zg_templates.add ((ev as GLib.Object).ref());
    }

    private string prepare_search_string (Search? search, Section section)
    {
      
      string s;
      if (search != null)
        s = search.get_search_string ();
      else
        s = "";

      s = s.strip ();
      
      if (!s.has_suffix ("*") && s != "")
        s = s + "*";
      
      if (s != "")
        s = @"app:($s)";
      else
        return section_queries.get(section);
      
      if (section == Section.ALL_APPLICATIONS)
        return s;
      else
        return s + @" AND $(section_queries.get(section))";
    }

    /* Returns true iff new_search is a query that simply restricts the
     * result set of old_query */
    private bool check_is_filter_search (Search new_search, Search? old_search)
    {
      bool is_filter_search = false;      
      if (old_search != null)
        {
          string previous_search_string = old_search.get_search_string();
          if (previous_search_string != null && previous_search_string != "")
            is_filter_search = new_search.get_search_string().has_prefix (
                                                        previous_search_string);
        }
      
      return is_filter_search;
    }
    
    private bool check_search_invalid (Search? search)
    {
      if (search == null ||
          search.get_search_string () == null ||
          search.get_search_string () == "")
        return true;
      return false;
    }

    private async void search_async (Search search,
                                     Section section,
                                     Dee.Model results_model,
                                     Dee.Model groups_model,
                                     bool is_filter_search,
                                     Dee.Index results_by_group)
    { 
      var search_string = prepare_search_string (search, section);

      try {
        var results = yield zg_index.search (search_string,
                                          new Zeitgeist.TimeRange.anytime(),
                                          zg_templates,
                                          Zeitgeist.StorageState.ANY,
                                          20,
                                          Zeitgeist.ResultType.MOST_POPULAR_SUBJECTS,
                                          null);

        if (!is_filter_search)
          {
            results_model.clear ();
            append_events_with_group (results, results_model,
                                      Group.MOST_USED, section);
          }
        else
          {
            debug ("Doing filter search on Most Used apps: '%s'",
                   search.get_search_string ());
            uint group = Group.MOST_USED;
            Dee.ResultSet filter_set = results_by_group.lookup (@"$group",
                                                                TermMatchFlag.EXACT);
            
            /* Build a set with all event actors */
            Set<string> app_uris = new HashSet<string> ();
            foreach (Event ev in results)
              {
                if (ev.num_subjects () > 0)
                  app_uris.add (ev.get_subject (0).get_uri ());
              }
                                                                
            Utils.apply_uri_filter(app_uris, filter_set);
          }
          
        debug ("Found %u/%u Most Used apps for query '%s'",
               results.size (), results.estimated_matches (), search_string);

      } catch (GLib.Error e) {
        warning ("Error performing search '%s': %s",
                 search.get_search_string (), e.message);
      }
      
      // FIXME: We could parallelize the queries to the packages and zg
      update_pkg_search (search, section, results_model,
                         is_filter_search, results_by_group);
      
      if (check_search_invalid (search))
        check_empty_section (section, results_model);
      else
        check_empty_search (search, results_model);
    }
    
    private string prepare_pkg_search_string (Search? search, Section section)
    {       
      if (check_search_invalid (search))
        {
          if (section == Section.ALL_APPLICATIONS)
            return "type:Application";
          else
            return @"type:Application AND $(section_queries.get(section))";
        }
      else
        {
          var s = search.get_search_string ();

          s = s.strip ();

          if (!s.has_suffix ("*"))
            s = s + "*";

          if (section == Section.ALL_APPLICATIONS)
            return @"($s) AND type:Application";
          else
            return @"($s) AND type:Application AND $(section_queries.get(section))";
        }
    }
    
    private Icon find_pkg_icon (Unity.Package.PackageInfo pkginfo)
    {
      string desktop_id = Path.get_basename (pkginfo.desktop_file);
      bool installed = AppInfoManager.get_instance().lookup (desktop_id) != null;
      
      /* If the app is already installed we should be able to pull the
       * icon from the theme */
      if (installed)
        return new ThemedIcon (pkginfo.icon);
      
      /* App is not installed - we need to find the right icon in the bowels
       * of the software center */
      if (pkginfo.icon.has_prefix ("/"))
        {
          return new FileIcon (File.new_for_path (pkginfo.icon));
        }
      else
        {
          Icon icon = file_icon_cache.lookup (pkginfo.icon);
          
          if (icon != null)
            return icon;
        
          foreach (var ext in image_extensions)
          {
            string path = @"$(Config.DATADIR)/app-install/icons/$(pkginfo.icon).$(ext)";
            File f = File.new_for_path (path);
            if (f.query_exists (null))
              {
                /* Got it! Cache the icon path and return the icon */
                icon = new FileIcon (f);
                file_icon_cache.insert (pkginfo.icon, icon);
                return icon;
              }
          }
        }
      
      /* Cache the fact that we couldn't find this icon */
      var icon = new ThemedIcon ("applications-other");
      file_icon_cache.insert (pkginfo.icon, icon);
      
      return icon;
    }

    private void on_appinfo_changed (string id, AppInfo? appinfo)
    {
      debug ("Application changed: %s", id);
      //update_entry_results_model.begin ();
    }

    private bool search_is_invalid (Search? search)
    {
      /* This boolean expression is unfolded as we seem to get
       * some null dereference if we join them in a big || expression */
      if (search == null)
        return true;
      else if (search.get_search_string () == null)
        return true;
      
      return search.get_search_string () == "";
    }

    private bool update_pkg_search (Search? search,
                                    Section section,
                                    Dee.Model model,
                                    bool is_filter_search,
                                    Dee.Index results_by_group)
    {      
      string search_string = prepare_pkg_search_string (search, section);
      bool has_search = !search_is_invalid (search);
      
      Timer timer = new Timer ();
      var appresults = appsearcher.search (search_string);     
      Set<string> installed_uris = new HashSet<string> ();
      Set<string> available_uris = new HashSet<string> ();
      
      add_pkg_search_result (appresults, installed_uris, available_uris, model,
                             Group.INSTALLED, results_by_group, is_filter_search);
      
      timer.stop ();
      debug ("Listed %i Installed apps in %fms for query: %s",
             appresults.num_hits, timer.elapsed ()*1000, search_string);
      
      /* If we have no search, don't display non-installed apps */
      if (has_search)
        {
          timer.start ();
          var pkgresults = pkgsearcher.search (search_string);
          add_pkg_search_result (pkgresults, installed_uris, available_uris,
                                 model, Group.AVAILABLE, results_by_group,
                                 is_filter_search);
          timer.stop ();
          debug ("Listed %i Available apps in %fms for query: %s",
                 pkgresults.num_hits, timer.elapsed ()*1000, search_string);
        }
      
      return false;
    }
    
    private void add_pkg_search_result (Unity.Package.SearchResult results,
                                        Set<string> installed_uris,
                                        Set<string> available_uris,
                                        Dee.Model model,
                                        Group group,
                                        Dee.Index results_by_group,
                                        bool is_filter_search)
    {
      var appmanager = AppInfoManager.get_instance();
    
      foreach (var pkginfo in results.results)
      {
      	if (pkginfo.desktop_file == null)
          continue;
                
        string desktop_id = Path.get_basename (pkginfo.desktop_file);
        AppInfo? app = appmanager.lookup (desktop_id);        
        
        /* De-dupe by 'application://foo.desktop' URI. Also note that we need
         * to de-dupe before we chuck out NoDisplay app infos, otherwise they'd
         * show up from alternate sources */
        string uri = @"application://$(desktop_id)";
        if (uri in installed_uris || uri in available_uris)
          continue;
        
        /* Extract basic metadata and register de-dupe keys */
        string display_name;
        string comment;
        switch (group)
        {
          case Group.INSTALLED:
            installed_uris.add (uri);
            display_name = app.get_display_name ();
            comment = app.get_description ();
            break;
          case Group.AVAILABLE:
            available_uris.add (uri);
            display_name = pkginfo.application_name;
            comment = "";
            break;
          default:
            warning (@"Illegal group for package search $(group)");
            continue;
        } 
        
        /* We can only chuck out NoDisplay and OnlyShowIn app infos after
         * we have registered a de-dupe key for them - which is done in the
         * switch block above) */
        if (app != null && !app.should_show ())
          continue;
                
        if (group == Group.AVAILABLE)
          {
            /* If we have an available item, which is not a dupe, but is
             * installed anyway, we weed it out here, because it's probably
             * left out from the Installed section because of some rule in the
             * .menu file */
            if (app != null)
              continue;            
            
            /* Apps that are not installed, ie. in the Available group
             * use the 'unity-install://pkgname/Full App Name' URI scheme,
             * but only use that after we've de-duped the results.
             * But only change the URI *after* we've de-duped the results! */
            uri = @"unity-install://$(pkginfo.package_name)/$(pkginfo.application_name)";
            available_uris.add (uri);            
          }
        
        Icon icon = find_pkg_icon (pkginfo);
        
        if (!is_filter_search)
          {
            model.append (ResultsColumn.URI, uri,
                          ResultsColumn.ICON_HINT, icon.to_string (),
                          ResultsColumn.GROUP_ID, group,
                          ResultsColumn.MIMETYPE, "application/x-desktop",
                          ResultsColumn.DISPLAY_NAME, display_name,
                          ResultsColumn.COMMENT, comment,
                          -1);
          }
      }
      
      if (is_filter_search)
        {
          /* Filter the Installed group */
          var filter_set = results_by_group.lookup (@"$((uint)group)",
                                                    TermMatchFlag.EXACT);
          switch (group)
          {
            case Group.INSTALLED:
              Utils.apply_uri_filter (installed_uris, filter_set);
              break;
            case Group.AVAILABLE:
              Utils.apply_uri_filter (available_uris, filter_set);
              break;
            default:
              warning (@"Illegal group for package search $(group)");
              break;
          }
        }
    }
    
    /**
     * Override of the default activation handler. The apps place daemon
     * can handle activation of installable apps using the Software Center
     */
    public async uint32 activate (string uri)
    {      
      if (!uri.has_prefix ("unity-install://"))
        {
          debug ("Declined activation of URI '%s': Expected URI scheme unity-install://", uri);
          return ActivationStatus.NOT_ACTIVATED;
        }
      
      unowned string pkg = uri.offset (16); // strip off "unity-install://" prefix
      debug ("Installing: %s", pkg);
      string[] args = new string[2];
      args[0] = "software-center";
      args[1] = pkg;

      try {
        Process.spawn_async (null, args, null, SpawnFlags.SEARCH_PATH, null, null);
      } catch (SpawnError e) {
        warning ("Failed to spawn software-center for URI '%s': %s",
                 uri, e.message);
        return ActivationStatus.NOT_ACTIVATED;
      }

      return ActivationStatus.ACTIVATED_HIDE_DASH;
      
    }
    
    /* Returns the GMenu.TreeDir for the matching section or null on error */
    /*private GMenu.TreeDirectory? get_directory_for_section (Section section)
      {  
        if (section >= Section.LAST_SECTION)
          {
            warning ("Illegal section requested: %u", section);
            return null;
          }

        var root = app_menu.get_root_directory ();
        string section_name = Utils.get_section_name (section);

        foreach (TreeItem item in root.get_contents ())
          {
            if (item.get_type () == TreeItemType.DIRECTORY)
              {
                unowned TreeDirectory treedir = (TreeDirectory)item;
                if (section_name == treedir.get_menu_id ())
                  return treedir;
              }

            if (item.get_type () == TreeItemType.ENTRY)
              {
                unowned TreeEntry entry = (TreeEntry)item;
                warning ("Unexpected entry in root menu: %s",
                         entry.get_desktop_file_path ());
              }
          }

        warning ("Unable to find menu node for section: %u", section);
        return null;
      }*/

    /* Appends the subject URIs from a set of Zeitgeist.Events to our Dee.Model
     * assuming that these events are already sorted */
    public void append_events_with_group (Zeitgeist.ResultSet events,
                                          Dee.Model results,
                                          uint group_id,
                                          int section_filter = -1)
    {
      foreach (var ev in events)
        {
          string app_uri;
          if (ev.num_subjects () > 0)
            app_uri = ev.get_subject (0).get_uri ();
          else
            {
              warning ("Unexpected event without subject");
              continue;
            }
        
          /* Assert that we indeed have a known application as actor */
          AppInfo? app = Utils.get_app_info_for_actor (app_uri);
          
          if (app == null)
            continue;
          
          if (!app.should_show ())
            continue;          
          
          results.append (ResultsColumn.URI, app_uri,
                          ResultsColumn.ICON_HINT, app.get_icon().to_string(),
                          ResultsColumn.GROUP_ID, group_id,
                          ResultsColumn.MIMETYPE, "application/x-desktop",
                          ResultsColumn.DISPLAY_NAME, app.get_display_name (),
                          ResultsColumn.COMMENT, app.get_description (),
                          -1);
        }
    }
    
    public void check_empty_search (Search? search,
                                    Dee.Model results_model)
    {
      if (results_model.get_n_rows () > 0)
        return;
      
      if (search_is_invalid(search))
        return;
      
      results_model.append (ResultsColumn.URI, "",
                            ResultsColumn.ICON_HINT, "",
                            ResultsColumn.GROUP_ID, Group.EMPTY_SEARCH,
                            ResultsColumn.MIMETYPE, "",
                            ResultsColumn.DISPLAY_NAME, _("Your search did not match any applications"),
                            ResultsColumn.COMMENT, "",
                            -1);      
      
      // FIXME: Use prefered browser
      // FIXME: URL escape search string
      results_model.append (ResultsColumn.URI, @"http://google.com/#q=$(search.get_search_string())",
                            ResultsColumn.ICON_HINT, "",
                            ResultsColumn.GROUP_ID, Group.EMPTY_SEARCH,
                            ResultsColumn.MIMETYPE, "",
                            ResultsColumn.DISPLAY_NAME, _("Search the web"),
                            ResultsColumn.COMMENT, "",
                            -1);
    }
    
    public void check_empty_section (Section section,
                                     Dee.Model results_model)
    {
      if (results_model.get_n_rows () > 0)
        return;
            
      string section_name;
      
      switch (section)
      {
        case Section.ALL_APPLICATIONS:
          section_name = _("applications");
          break;
        case Section.ACCESSORIES:
          section_name = _("accessories");
          break;
        case Section.GAMES:
          section_name = _("games");
          break;
        case Section.INTERNET:
          section_name = _("internet applications");
          break;
        case Section.MEDIA:
          section_name = _("media applications");
          break;
        case Section.OFFICE:
          section_name = _("office applications");
          break;
        case Section.SYSTEM:
          section_name = _("system applications");
          break;
        default:
          section_name = _("applications");
          warning ("Unknown section: %u", section);
          break;
      }
      
      results_model.append (ResultsColumn.URI, "",
                            ResultsColumn.ICON_HINT, "",
                            ResultsColumn.GROUP_ID, Group.EMPTY_SECTION,
                            ResultsColumn.MIMETYPE, "",
                            /* TRANSLATORS: The %s is plural. Eg. "games" */
                            ResultsColumn.DISPLAY_NAME, _("There are no %s installed on this computer").printf (section_name),
                            ResultsColumn.COMMENT, "",
                            -1);
    }
  
  } /* END: class Daemon */
  
  /*public void append_treedir_with_group (GMenu.TreeDirectory treedir,
                                        Dee.Model results,
                                        uint group_id)
  {
      foreach (var item in treedir.get_contents())
        {
          if (item.get_type () == TreeItemType.DIRECTORY)
          {
            warning ("Nested tree in treedir. Internal error");
            continue;
          }
          
          unowned TreeEntry entry = (TreeEntry)item;
          
          if (entry.get_is_nodisplay ())
            continue;
          
          string app_uri = @"application://$(entry.get_desktop_file_id())";
          results.append (ResultsColumn.URI, app_uri,
                          ResultsColumn.ICON_HINT, entry.get_icon(),
                          ResultsColumn.GROUP_ID, group_id,
                          ResultsColumn.MIMETYPE, "application/x-desktop",
                          ResultsColumn.DISPLAY_NAME, entry.get_name(),
                          ResultsColumn.COMMENT, entry.get_comment (),
                          -1);
          debug ("APP: %s", app_uri);
        }
  }*/

} /* namespace */
