/*****************************************************************************
 *
 * grail - Gesture Recognition And Instantiation Library
 *
 * Copyright (C) 2010 Canonical Ltd.
 * Copyright (C) 2010 Henrik Rydberg <rydberg@bitmath.org>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include <grail-touch.h>
#include <errno.h>
#include <math.h>

/* see mtdev-mapping.h */
#define MTDEV_POSITION_X	5
#define MTDEV_POSITION_Y	6
#define MTDEV_ORIENTATION	4
#define MTDEV_PRESSURE		10

#define SYSCALL(call) while (((call) == -1) && (errno == EINTR))

static const int bits_per_long = 8 * sizeof(long);

static inline int nlongs(int nbit)
{
	return (nbit + bits_per_long - 1) / bits_per_long;
}

static inline int getbit(const unsigned long *map, int key)
{
	return (map[key / bits_per_long] >> (key % bits_per_long)) & 0x01;
}

int touch_caps_is_supported(struct touch_dev *dev, int fd)
{
	unsigned long keybits[nlongs(KEY_MAX)];
	unsigned long absbits[nlongs(ABS_MAX)];
	int rc;

	if (dev->mtdev.caps.has_mtdata)
		return 1;

	SYSCALL(rc = ioctl(fd, EVIOCGBIT(EV_KEY, sizeof(keybits)), keybits));
	if (rc < 0)
		return 0;
	SYSCALL(rc = ioctl(fd, EVIOCGBIT(EV_ABS, sizeof(absbits)), absbits));
	if (rc < 0)
		return 0;

	return getbit(absbits, ABS_X) &&
		getbit(absbits, ABS_Y) &&
		getbit(keybits, BTN_TOUCH) &&
		getbit(keybits, BTN_TOOL_DOUBLETAP);
}

void touch_caps_init(struct touch_dev *dev)
{
	const struct mtdev_caps *mt = &dev->mtdev.caps;
	struct touch_caps *caps = &dev->caps;
	caps->min_x = mt->abs[MTDEV_POSITION_X].minimum;
	caps->max_x = mt->abs[MTDEV_POSITION_X].maximum;
	caps->min_y = mt->abs[MTDEV_POSITION_Y].minimum;
	caps->max_y = mt->abs[MTDEV_POSITION_Y].maximum;
	caps->min_orient = mt->abs[MTDEV_ORIENTATION].minimum;
	caps->max_orient = mt->abs[MTDEV_ORIENTATION].maximum;
	caps->min_press = mt->abs[MTDEV_PRESSURE].minimum;
	caps->max_press = mt->abs[MTDEV_PRESSURE].maximum;
	if (caps->min_x == caps->max_x) {
		caps->min_x = 0;
		caps->max_x = 1024;
	}
	if (caps->min_y == caps->max_y) {
		caps->min_y = 0;
		caps->max_y = 768;
	}
	if (caps->min_orient == caps->max_orient)
		caps->max_orient = 1;
	if (caps->min_orient + caps->max_orient != 0)
		caps->min_orient = -caps->max_orient;
	if (caps->min_press == caps->max_press) {
		caps->min_press = 0;
		caps->max_press = 256;
	}
}

float touch_angle(const struct touch_dev *dev, int orient)
{
	const struct touch_caps *caps = &dev->caps;
	return orient / caps->max_orient * M_PI_2;
}

float touch_pressure(const struct touch_dev *dev, int press)
{
	const struct touch_caps *caps = &dev->caps;
	return (press - caps->min_press) / (caps->max_press - caps->min_press);
}
